$version = "2024-08-28-0_KR"

function Write-OK { param($str) Write-Host -ForegroundColor green $str } 
function Write-nonOK { param($str) Write-Host -ForegroundColor red $str } 
function Write-Warning { param($str) Write-Host -ForegroundColor magenta $str } 
function Write-Info { param($str) Write-Host -ForegroundColor yellow $str } 
function Write-Detail { param($str) Write-Host -ForegroundColor gray $str } 
function Write-Break { Write-Host -ForegroundColor Gray "================================================================================" }
function Write-Title 
{
    param($str)
    Write-Host -ForegroundColor Yellow ("=" * ($str.Length + 4))
    Write-Host -ForegroundColor Yellow "| $str |" 
    Write-Host -ForegroundColor Yellow ("=" * ($str.Length + 4))
} 

function GetRegSetValueString
{
    param ($path, $propName, $value)
    $path = $path.replace("HKLM:\", "HKEY_LOCAL_MACHINE\")
    return "[microsoft.win32.registry]::SetValue(""$path"", ""$propName"", $value)"
}

function OutputMitigationToPs1 
{
    param ($mitigationId, $script, $printDone = $true)
    $fileName = ".\Mitigation-$mitigationId.ps1"
    $cmt = "# This PowerShell script was generated as a mitigation by Azure DevOps TLS 1.2 transition readiness checker."
    $lines = @($cmt) + $script
    if ($printDone) { $lines += @("'Done!'") }
    $lines | Out-File -FilePath $fileName -Force
    $filePathAbsolute = $fileName | Resolve-Path
    return $filePathAbsolute
}

# Define a function to get the .NET Framework version (added by KR)
function Get-DotNetFrameworkVersion {
    $registryPath = "HKLM:\SOFTWARE\Microsoft\NET Framework Setup\NDP\v4\Full"

    if (Test-Path $registryPath) {
        $releaseKey = (Get-ItemProperty -Path $registryPath -Name Release -ErrorAction SilentlyContinue).Release
        if ($releaseKey) {
            $dotNetFrameworkVersions = @{
                378389 = "4.5"
                378675 = "4.5.1"
                378758 = "4.5.1"
                379893 = "4.5.2"
                393295 = "4.6"
                393297 = "4.6"
                394254 = "4.6.1"
                394271 = "4.6.1"
                394802 = "4.6.2"
                394806 = "4.6.2"
                460798 = "4.7"
                460805 = "4.7"
                461308 = "4.7.1"
                461310 = "4.7.1"
                461808 = "4.7.2"
                461814 = "4.7.2"
                528040 = "4.8"   # Windows 10 May 2019 Update and Windows 10 November 2019 Update
                528049 = "4.8"   # all other OS versions
                528372 = "4.8"   # Windows 10 May 2020 Update and Windows 10 October 2020 Update and Windows 10 May 2021 Update
                528449 = "4.8"   # Windows 11 and Windows Server 2022
				533320 = "4.8.1" # Windows 11 September 2022 Release and Windows 11 October 2023 Release
                533325 = "4.8.1" # all other OS versions
            }

            if ($dotNetFrameworkVersions.ContainsKey($releaseKey)) {
                return "The .NET Framework version is $($dotNetFrameworkVersions[$releaseKey]) (Release key: $releaseKey)"
            } else {
                return "Unknown .NET Framework version (Release key: $releaseKey)"
            }
        } else {
            return "The .NET Framework Release key is not found."
        }
    } else {
        return ".NET Framework v4.0 or later is not installed."
    }
}

#
#
#
Write-Title "Analysis of TLS 1.2 compatibility: OS"
#
#
#

Write-Detail "Getting environment info..."
Write-Break

$envOsVersion = [System.Environment]::OSVersion.Version # if OS went through update (W8 -> W8.1 -> W10 ...), this may return pre-update version (https://stackoverflow.com/questions/33328739/system-environment-osversion-returns-wrong-version) 
$winVersionRex = "([0-9]+\.)+[0-9]+"
$systemInfoVersion = $null
if ((systeminfo /fo csv | ConvertFrom-Csv | Select-Object -Property "OS Version")."OS Version" -match $winVersionRex) { $systemInfoVersion = [version]$Matches[0] } # systeminfo command is considered obsolete but gives up to date version
$osVersion = if ($envOsVersion -gt $systemInfoVersion) { $envOsVersion } else { $systemInfoVersion } # Take the highest OS version seen

Write-Host "PS Version:" $PSversionTable.PSVersion
Write-Host "PS Edition: " $PSversionTable.PSEdition
Write-Host "CLR Version: " $PSversionTable.CLRVersion
Write-Host "OS Version: system.environment: $envOsVersion, systeminfo: $systemInfoVersion --> $osVersion"

Write-Break


Write-Detail "Running Hot Fix check..."

function CheckHotfix {
    param ($hotfixId)
    
    Write-Detail "Checking $hotfixId..."
    $hotfix = Get-HotFix | Where-Object { $_.HotFixID -eq $hotfixId } 
    if ($hotfix)
    {
        Write-Detail "Hotfix $hotfixId found: $hotfix"
        return $true
    }
    else
    {
        Write-Detail "Hotfix $hotfixId not installed."
        return $false
    }
}

if ($osVersion.Major -ge 10)
{
    Write-OK "No hot fixes are necessary for TLS 1.2 support on this OS version."
}
elseif ($osVersion -ge [version]"6.3")
{
    $hotfixId = "KB2919355"
    $found = CheckHotfix $hotfixId    
    if ($found)
    {
        Write-OK "Hotfix check passed."
    }
    else
    {
        Write-nonOK "ISSUE FOUND: $hotfixId missing, see https://learn.microsoft.com/en-us/windows/win32/secauthn/tls-cipher-suites-in-windows-8-1"
    }
}
elseif ($osVersion -ge [version]"6.1")
{
    $hotfixId = "KB3140245"
    $found = CheckHotfix $hotfixId    
    if ($found)
    {
        Write-OK "Hotfix check passed."
    }
    else
    {
        Write-nonOK "ISSUE FOUND: $hotfixId missing, see https://support.microsoft.com/en-us/topic/update-to-enable-tls-1-1-and-tls-1-2-as-default-secure-protocols-in-winhttp-in-windows-c4bd73d2-31d7-761e-0178-11268bb10392"
    }
}
else
{
    Write-Error "This version of OS is not supported by the troubleshooting script (supported: Windows Server 2008 R2+, Windows 7+)"
}

# Define the registry paths and keys with their desired values
$registrySettings = @{
    "HKLM:\SYSTEM\CurrentControlSet\Control\SecurityProviders\SCHANNEL\Protocols\TLS 1.2\Server" = @{
        "Enabled" = 1
        "DisabledByDefault" = 0
    }
    "HKLM:\SYSTEM\CurrentControlSet\Control\SecurityProviders\SCHANNEL\Protocols\TLS 1.2\Client" = @{
        "Enabled" = 1
        "DisabledByDefault" = 0
    }
    "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Internet Settings\WinHttp" = @{
        "DefaultSecureProtocols"= 2560
    }
}

# Function to check and set registry keys
function CheckAndSet-RegistryKeys {
    param (
        [string]$regPath,
        [hashtable]$keysToSet
    )

    # Check if the registry path exists, if not, create it
    if (-not (Test-Path -Path $regPath)) {
        New-Item -Path $regPath -Force | Out-Null
        Write-Warning "Registry path created: $regPath"
    } else {
        Write-OK "Registry path already exists: $regPath"
    }

    # Loop through each key-value pair and ensure it is set correctly
    foreach ($key in $keysToSet.Keys) {
        $currentValue = (Get-ItemProperty -Path $regPath -Name $key -ErrorAction SilentlyContinue).$key

        if ($currentValue -ne $keysToSet[$key]) {
            Set-ItemProperty -Path $regPath -Name $key -Value $keysToSet[$key] -Force
            Write-Warning "Set $key to $($keysToSet[$key]) in $regPath"
        } else {
            Write-OK "$key is already set to $($keysToSet[$key]) in $regPath"
        }
    }
    Write-Break
}

Write-Break

# Iterate through the registry settings and apply them
foreach ($regPath in $registrySettings.Keys) {
    CheckAndSet-RegistryKeys -regPath $regPath -keysToSet $registrySettings[$regPath]
}

#
#
#
Write-Title "Analysis of TLS 1.2 compatibility: .NET Framework"
#
#
#

$netFwkVersionPath = "HKLM:\SOFTWARE\Microsoft\NET Framework Setup\NDP\v4\Full"
# Mapping table from the above Release number to .NET version here: 
# https://learn.microsoft.com/en-us/dotnet/framework/migration-guide/how-to-determine-which-versions-are-installed#detect-net-framework-45-and-later-versions
if (Test-Path -Path $netFwkVersionPath)
{
    $fwkRelease = (Get-ItemProperty -Path $netFwkVersionPath).Release
    if ($fwkRelease -lt 460000)
    {
        Write-nonOK ".NET Framework 4.7+ not installed (release $fwkRelease). This may cause TLS-compatibility issues to .NET applications."
    }
    else
    {
        Write-OK ".NET Framework release is 4.7+ (release $fwkRelease)"
    }
}
else
{
    Write-nonOK ".NET Framework 4.7+ not installed (version below 4.5 seem to be installed)"
}

function CheckRegistryDefined
{
    param($path, $property)
    if (-not (Test-Path -Path $path)) { return $null }
    return Get-ItemProperty -Path $path
}

function CheckStrongCrypto 
{
    param($path, $desc)
    $isStrongCryptoEnforced = 
        (($propertyObject = (CheckRegistryDefined $path)) -and
        (($null -ne $propertyObject.SchUseStrongCrypto) -and ($propertyObject.SchUseStrongCrypto -ne 0)) -and
        (($null -ne $propertyObject.SystemDefaultTlsVersions) -and ($propertyObject.SystemDefaultTlsVersions -ne 0)))

    if ($isStrongCryptoEnforced)
    {
        Write-OK "TLS 1.2 enforced for applications targetting $desc"
    }
    else
    {
        Write-Warning "Warning: TLS 1.2 not enforced for applications targetting $desc"
        $result = @(
            (GetRegSetValueString $path "SchUseStrongCrypto" 1),
            (GetRegSetValueString $path "SystemDefaultTlsVersions" 1)
        )
        return $result
    }
}

$mitigations = @()
$mitigations = $mitigations + (CheckStrongCrypto "HKLM:\SOFTWARE\Microsoft\.NETFramework\v4.0.30319" ".NET Framework 4.0/4.5.x")
$mitigations = $mitigations + (CheckStrongCrypto "HKLM:\SOFTWARE\WOW6432Node\Microsoft\.NETFramework\v4.0.30319" ".NET Framework 4.0/4.5.x (32bit app on 64bit OS)")
$mitigations = $mitigations + (CheckStrongCrypto "HKLM:\SOFTWARE\Microsoft\.NETFramework\v2.0.50727" ".NET Framework 3.5")
$mitigations = $mitigations + (CheckStrongCrypto "HKLM:\SOFTWARE\Wow6432Node\Microsoft\.NETFramework\v2.0.50727" ".NET Framework 3.5 (32bit app on 64bit OS)")
$mitigations = $mitigations | Where-Object { $_ -ne $null } 
if ($mitigations.Count -gt 0)
{
    $scriptFile = OutputMitigationToPs1 "NetFramework" $mitigations
    
    Write-Info "Follow the below mitigations when the OS analysis is without issues and there are still applications with TLS-connectivity issues on the computer."
    Write-Warning "MITIGATIONS: per https://learn.microsoft.com/en-us/mem/configmgr/core/plan-design/security/enable-tls-1-2-client"    
    Write-Warning "    Mitigation script generated at $scriptFile"
    Write-Warning "    Run the mitigation script as Administrator and restart the computer."
}
else
{
    Write-OK "All mitigations required to ensure TLS 1.2-compatibility of legacy .NET applications are in place."
}

# Get and display the .NET Framework version
$dotNetFrameworkVersion = Get-DotNetFrameworkVersion
Write-Output $dotNetFrameworkVersion
